unit Horizon01;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, ComCtrls,
  Math;

type THorizon = class(Tobject)
private
    // ---------------
    fPanel      : TPanel;       //   
    fImg        : TImage;
    fXB         : integer;
    fYB         : integer;
    fRect       : TRect;
    // ---------------
    fTrackRoll  : TTrackBar;
    fSTxtRoll   : TStaticText;
    // ---------------
    fTrackPitch : TTrackBar;
    fSTxtPitch  : TStaticText;
    // ---------------
    fLbRoll     : TLabel;
    fLbPitch    : TLabel;
    // ---------------
    //      
    fPolarArr   : array of record
       L : extended;  //  
       A : extended;  //   
    end;
    //      
    fAirCraft : array of TPoint;
    // ---------------
    //     
    procedure TrackRollChange(Sender: TObject);
    //     
    procedure TrackPitchChange(Sender: TObject);
    // ---------------
    //      
    procedure DsAirCraftToPolar();
    //   
    procedure ClearRoll();
    //   
    function ClearPitch() : TRect;
    //   
    procedure ShowAxis();
    //      
    procedure ShowRollMarks();
    //  
    procedure ShowPitch(RqPitch : extended);
    //      
    procedure ShowAirCraft(RqAngle : extended);
    //   
    procedure ShowHorizon();
    // ---------------
    procedure SetRoll(RqRoll : integer);
    function  GetRoll() : integer;
    procedure SetPitch(RqPitch : integer);
    function  GetPitch() : integer;
    // ---------------
public
    constructor Create(
          RqWinControl : TWinControl;  //      
          XB, YB : integer);
    procedure Free();
    // ---------------
    //     (-90 .. 90 )
    property Roll : integer read GetRoll write SetRoll;
    //     (-89 .. 89 )
    property Pitch : integer read GetPitch write SetPitch;
end;

// =========================================================================
// =========================================================================

implementation

// =========================================================================
// =========================================================================
//       
const Radius1 = 200;       //   
      XZero   = Radius1;   // Left -  
      YZero   = Radius1;   // Top  -  
// ----------------------------------------------------------------
//      
const MaxIndAirCraft = 12;
const DsAirCraft : array[0..MaxIndAirCraft] of TPoint =
(
  (X : - Radius1;        Y :  0),
  (X : - Radius1 + 30;   Y :  4),
  (X : - 30;             Y :  4),
  (X :    0;             Y : -20),
  (X : + 30;             Y :  4),
  (X : Radius1 - 30;     Y :  4),
  (X : Radius1;          Y :  0),
  (X : Radius1 - 30;     Y : -4),
  (X : + 30;             Y : -4),
  (X :    0;             Y : -28),
  (X : - 30;             Y : -4),
  (X : - Radius1 + 30;   Y : -4),
  (X : - Radius1;        Y :  0)
);

// ================================================================
//     
// ================================================================
// ----------------------------------------------------------------
// ----------------------------------------------------------------
//   ,   TControl.
//  CreateControl    class () 
// ,      RqOwner
// (,  Form1).

function CreateControl(ControlClass  : TControlClass;
                       RqOwner       : TWinControl;
                 const ComponentName : string;
                       X, Y, W, H    : Integer): TControl;
begin
    Result := ControlClass.Create(RqOwner);
    with Result do
    begin
      Parent := RqOwner;               //    
      Name := ComponentName;           //   
      SetBounds(X, Y, W, H);           //   TWinControl
      Visible := True;                 //   
    end;
end;
// ================================================================
//    / 
// ================================================================
// ----------------------------------------------------------------
constructor THorizon.Create(
            RqWinControl : TWinControl;  //      
            XB, YB : integer);
begin
   inherited Create;
   // ----------------------------------------------
   //      
   fPanel:= TPanel(CreateControl(TPanel, RqWinControl, '',
                   XB,YB,512,512));
   // ----------------------------------------------
   fImg := TImage(CreateControl(TImage, fPanel, '',
                  4,4,440,440));
   fXB  := 20;
   fYB  := 20;
   fRect := Rect(fXB,fYB, fXB + 2 * Radius1,fYB + 2 * Radius1);
   // ----------------------------------------------
   //  
   fTrackRoll := TTrackBar(CreateControl(TTrackBar, fPanel, '',
                           50, 470, 440, 32));
   fTrackRoll.Min := -90;
   fTrackRoll.Max :=  90;
   fTrackRoll.Position := 0;
   fTrackRoll.OnChange := TrackRollChange;
   // ----------------------------------------------
   //   
   fSTxtRoll := TStaticText(CreateControl(TStaticText, fPanel, '',
                            10, 472, 40, 16));
   fSTxtRoll.Alignment   := taCenter;
   fSTxtRoll.AutoSize    := False;
   fSTxtRoll.BorderStyle := sbsSunken;
   fSTxtRoll.Font.Style  := [fsBold];
   fSTxtRoll.Caption := '0';
   // ----------------------------------------------
   //  
   fTrackPitch := TTrackBar(CreateControl(TTrackBar, fPanel, '',
                            464, 32, 32, 410));
   fTrackPitch.Orientation := trVertical;
   fTrackPitch.Width  := 32;
   fTrackPitch.Height := 410;
   fTrackPitch.Min := -88;
   fTrackPitch.Max :=  88;
   fTrackPitch.Position := 0;
   fTrackPitch.OnChange := TrackPitchChange;
   // ----------------------------------------------
   //   
   fSTxtPitch := TStaticText(CreateControl(TStaticText, fPanel, '',
                            464, 16, 40, 16));
   fSTxtPitch.Alignment   := taCenter;
   fSTxtPitch.AutoSize    := False;
   fSTxtPitch.BorderStyle := sbsSunken;
   fSTxtPitch.Font.Style  := [fsBold];
   fSTxtPitch.Caption := '0';
   // ----------------------------------------------
   fLbRoll := TLabel(CreateControl(TLabel, fPanel, '',
                     64,456,58,14));
   fLbRoll.AutoSize := True;
   fLbRoll.Caption := '   ';
   // ----------------------------------------------
   fLbPitch := TLabel(CreateControl(TLabel, fPanel, '',
                      456,440,38,14));
   fLbPitch.AutoSize := True;
   fLbPitch.Caption := '';
   // ----------------------------------------------
   // ----------------------------------------------
   //
   DsAirCraftToPolar();
   ShowHorizon();
end;
// ----------------------------------------------------------------
procedure THorizon.Free();
begin
    // ---------------
    SetLength(fPolarArr, 0);
    // ---------------
    fLbPitch.Free;
    fLbRoll.Free;
    // ---------------
    fSTxtPitch.Free;
    fTrackPitch.Free;
    fSTxtRoll.Free;
    fTrackRoll.Free;
    // ---------------
    fImg.Free;
    // ---------------
    fPanel.Free;
    // ---------------
    inherited Free;
end;

// ================================================================
//      
// ================================================================
// ----------------------------------------------------------------
//    
procedure THorizon.TrackRollChange(Sender: TObject);
begin
  fSTxtRoll.Caption := IntToStr(fTrackRoll.Position);
  ShowHorizon();
end;
// ----------------------------------------------------------------
//    
procedure THorizon.TrackPitchChange(Sender: TObject);
begin
  fSTxtPitch.Caption := IntToStr( -fTrackPitch.Position);
  ShowHorizon();
end;
// ================================================================
//    
// ================================================================
// ----------------------------------------------------------------
//      
procedure THorizon.DsAirCraftToPolar();
var wAngle : extended;
    Ind    : integer;
begin
    SetLength(fPolarArr, Length(DsAirCraft));
    for Ind := Low(fPolarArr) to High(fPolarArr)
    do begin
        //   
        fPolarArr[Ind].L := sqrt(DsAirCraft[Ind].X * DsAirCraft[Ind].X +
                                 DsAirCraft[Ind].Y * DsAirCraft[Ind].Y);
        //     
        if fPolarArr[Ind].L < 1e-14
        then wAngle := 0
        else wAngle := arcsin(Abs(DsAirCraft[Ind].Y) / fPolarArr[Ind].L);
        //     
        if (DsAirCraft[Ind].X >= 0 )
        then begin  // I  IV 
          if (DsAirCraft[Ind].Y >=0 )
          then fPolarArr[Ind].A :=    wAngle               // I  
          else fPolarArr[Ind].A :=  - wAngle;              // IV 
        end
        else begin  // II  III 
          if (DsAirCraft[Ind].Y >= 0 )
          then fPolarArr[Ind].A :=  Pi - wAngle            // II 
          else fPolarArr[Ind].A :=  Pi + wAngle;           // III 
        end;
    end;
end;
// ----------------------------------------------------------------
//   
procedure THorizon.ClearRoll();
begin
   with fImg.Canvas
   do begin
      Brush.Style := bsSolid;
      Brush.Color := clWindow;
      //  
      Ellipse(fRect);
   end;
end;
// ----------------------------------------------------------------
//   
function THorizon.ClearPitch() : TRect;
begin
    Result := Rect(fXB + 4, fYB + 4,
                   fXB + 2 * Radius1 - 4,
                   fYB + 2 * Radius1 - 4);
    with fImg.Canvas
    do begin
       Brush.Style := bsSolid;
      Brush.Color := clWindow;
       //  
       Ellipse(Result);
    end;
end;
// ----------------------------------------------------------------
procedure THorizon.ShowHorizon();
begin
    //   
    ClearRoll();
    //   
    ShowRollMarks();
    //  
    ShowPitch(fTrackPitch.Position);
    //  
    ShowAirCraft(fTrackRoll.Position);
end;
// ----------------------------------------------------------------
//      
procedure THorizon.ShowRollMarks();
const GdRd = Pi/180;
var wSin, wCos : extended;
    wPn1       : Tpoint;
    wAngle     : extended;
    Ind         : integer;
begin
    with fImg.Canvas
    do begin
       Pen.Width := 1;
       Pen.Color := clBlack;
       Font.Color := clBlack;
    end;
    //   
    Ind := 0;
    repeat
       wAngle := Ind * GdRd;
       SinCos(wAngle, wSin, wCos);
       wPn1.X := Round(Radius1 * wCos);
       wPn1.Y := Round(Radius1 * wSin);
       with fImg.Canvas
       do begin
          if (Ind mod 10) = 0
          then Pen.Width := 3
          else Pen.Width := 1;
          //  
          Ellipse(fXB + Radius1 + wPn1.X - 2, fYB + Radius1 - (wPn1.Y - 2),
                  fXB + Radius1 + wPn1.X + 2, fYB + Radius1 - (wPn1.Y + 2));
       end;
       Ind := Ind + 2;
    until (Ind > 359);
end;
// ----------------------------------------------------------------
//  
procedure THorizon.ShowAxis();
const GdRd = Pi/180;
var wY, wYm, wYp, Ind : integer;
begin
   with fImg.Canvas
   do begin
       Pen.Width := 1;
       Pen.Color := clBlack;
       Brush.Style := bsClear;
       //  
       MoveTo(fXB, fYB + Radius1);
       LineTo(fXB + 2 * Radius1, fYB + Radius1);
       //  
       MoveTo(fXB + Radius1, fYB);
       LineTo(fXB + Radius1, fYB + 2 * Radius1);
       //   
       Ind := 2;
       repeat
          if Ind >= 50
          then begin
            Pen.Color  := clRed;
            Font.Color := clRed;
          end;
          wY  := Round((Radius1 - 4) * Sin(Ind * GdRd));
          wYm := fYB + Radius1  - wY;
          wYp := fYB + Radius1  + wY;
          if Ind mod 10 <> 0
          then begin
            MoveTo(fXB + Radius1 - 2, wYm);
            LineTo(fXB + Radius1 + 2, wYm);
            MoveTo(fXB + Radius1 - 2, wYp);
            LineTo(fXB + Radius1 + 2, wYp);
          end
          else begin
            MoveTo(fXB + Radius1 - 4, wYm);
            LineTo(fXB + Radius1 + 4, wYm);

            TextOut(fXB + Radius1 + 6,
                    wYm - TextWidth('0') -1,
                    IntToStr(-Ind));
            MoveTo(fXB + Radius1 - 4, wYp);
            LineTo(fXB + Radius1 + 4, wYp);
            TextOut(fXB + Radius1 + 6,
                    wYp - TextWidth('0') -1,
                    '+' + IntToStr(Ind));
          end;
          Ind := Ind + 2;
       until (Ind > fTrackPitch.Max - 18);
   end;
end;

// ----------------------------------------------------------------
//  
procedure THorizon.ShowPitch(RqPitch : extended);
const GdRd = Pi/180;
var   wRect : TRect;
      wAngle1    : extended;
      wAngle2    : extended;
      wSin, wCos : extended;
      wPn1, wPn2 : Tpoint;
begin
     with fImg.Canvas
     do begin
        Pen.Width := 1;
        Pen.Color := clBlack;
        Font.Color := clBlack;
     end;
     if RqPitch >=0
     then begin                      //  (+)
         //   
         wAngle1 := Pi + (RqPitch * GdRd);         // III
         //   
         wAngle2 := - (RqPitch * GdRd);            // IV
     end
     else begin                     //  (-)
         //   
         wAngle1 := Pi - (- RqPitch) * GdRd;       // II
         //   
         wAngle2 := (- RqPitch) * GdRd;            // I
     end;
     //     
     SinCos(wAngle1, wSin, wCos);
     wPn1.X := Round(Radius1 * wCos);
     wPn1.Y := Round(Radius1 * wSin);
     //     
     SinCos(wAngle2, wSin, wCos);
     wPn2.X := Round(Radius1 * wCos);
     wPn2.Y := Round(Radius1 * wSin);
     //  
     with fImg.Canvas
     do begin
        //   
        wRect := ClearPitch();
        //   
        Brush.Color := RGB(200,250,200);
        //    
        Chord(wRect.Left, wRect.Top, wRect.Right, wRect.Bottom,
               fXB + XZero + wPn1.X,
               fYB + YZero - wPn1.Y,
               fXB + XZero + wPn2.X,
               fYB + YZero - wPn2.Y);
     end;
end;
// ----------------------------------------------------------------
//      
procedure THorizon.ShowAirCraft(RqAngle : extended);
const GdRd = Pi/180;
var wSin, wCos : extended;
    wAngle     : extended;
    wPn1       : Tpoint;
    Ind        : integer;
begin
    ShowAxis();
    SetLength(fAirCraft, Length(DsAirCraft));
    for Ind := Low(fAirCraft) to High(fAirCraft)
    do begin
        //    
        wAngle := RqAngle * GdRd + fPolarArr[Ind].A;
        //    
        SinCos(wAngle, wSin, wCos);
        wPn1.X := Round(fPolarArr[Ind].L * wCos);
        wPn1.Y := Round(fPolarArr[Ind].L * wSin);
        //    Polyline
        fAirCraft[Ind].X := fXB  + XZero + wPn1.X;
        fAirCraft[Ind].Y := fYB  + YZero - wPn1.Y;
    end;
    //      
    with fImg.Canvas
    do begin
      Pen.Width := 1;
      Pen.Color := clBlack;
      Font.Color := clBlack;
      Polyline(fAirCraft);
    end;
    SetLength(fAirCraft, 0);
end;
// ================================================================
//    PROPERTY
// ================================================================
// ----------------------------------------------------------------
//    
procedure THorizon.SetRoll(RqRoll : integer);
begin
   if (RqRoll >= fTrackRoll.Min) and (RqRoll <= fTrackRoll.Max)
   then begin
      fTrackRoll.Position := RqRoll;
      // ShowHorizon();
   end;
end;
// ----------------------------------------------------------------
//   
function  THorizon.GetRoll() : integer;
begin
   Result := fTrackRoll.Position;
end;
// ----------------------------------------------------------------
//    
procedure THorizon.SetPitch(RqPitch : integer);
begin
   if (RqPitch >= fTrackPitch.Min) and (RqPitch <= fTrackPitch.Max)
   then  begin
      fTrackPitch.Position := RqPitch;
      // ShowHorizon();
   end;
end;
// ----------------------------------------------------------------
//   
function  THorizon.GetPitch() : integer;
begin
   Result := - fTrackPitch.Position;
end;
// ================================================================
//   
// ================================================================

end.
